/*
 * SpanDSP - a series of DSP components for telephony
 *
 * v8_tests.c
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: v8_tests.c,v 1.19 2006/11/23 15:48:09 steveu Exp $
 */

/*! \page v8_tests_page V.8 tests
\section v8_tests_page_sec_1 What does it do?
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <inttypes.h>
#include <stdio.h>
#if defined(HAVE_TGMATH_H)
#include <tgmath.h>
#endif
#if defined(HAVE_MATH_H)
#include <math.h>
#endif
#include <fcntl.h>
#include <string.h>
#include <audiofile.h>
#include <tiffio.h>

#include "spandsp.h"

#define FALSE 0
#define TRUE (!FALSE)

#define SAMPLES_PER_CHUNK   160

#define OUTPUT_FILE_NAME    "v8.wav"

int negotiations_ok = 0;

static void handler(void *user_data, v8_result_t *result)
{
    const char *s;
    
    s = (const char *) user_data;
    
    if (result == NULL)
    {
        printf("%s V.8 negotiation failed\n", s);
        return;
    }
    printf("%s V.8 negotiation result:\n", s);
    printf("  Call function '%s'\n", v8_call_function_to_str(result->call_function));
    printf("  Negotiated modulation '%s'\n", v8_modulation_to_str(result->negotiated_modulation));
    printf("  Protocol '%s'\n", v8_protocol_to_str(result->protocol));
    printf("  PSTN access '%s'\n", v8_pstn_access_to_str(result->pstn_access));
    printf("  PCM modem availability '%s'\n", v8_pcm_modem_availability_to_str(result->pcm_modem_availability));
    if (result->call_function == V8_CALL_V_SERIES
        &&
        result->negotiated_modulation == V8_MOD_V90
        &&
        result->protocol == V8_PROTOCOL_LAPM_V42)
    {
        negotiations_ok++;
    }
}

int main(int argc, char *argv[])
{
    int i;
    int16_t amp[SAMPLES_PER_CHUNK];
    int16_t out_amp[2*SAMPLES_PER_CHUNK];
    v8_state_t v8_caller;
    v8_state_t v8_answerer;
    int outframes;
    int samples;
    int remnant;
    int caller_available_modulations;
    int answerer_available_modulations;
    AFfilehandle outhandle;
    AFfilesetup filesetup;
    
    if ((filesetup = afNewFileSetup()) == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }
    afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
    afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
    afInitFileFormat(filesetup, AF_FILE_WAVE);
    afInitChannels(filesetup, AF_DEFAULT_TRACK, 2);
    if ((outhandle = afOpenFile(OUTPUT_FILE_NAME, "w", filesetup)) == AF_NULL_FILEHANDLE)
    {
        fprintf(stderr, "    Cannot create wave file '%s'\n", OUTPUT_FILE_NAME);
        exit(2);
    }

    caller_available_modulations = V8_MOD_V17
                                 | V8_MOD_V21
                                 | V8_MOD_V22
                                 | V8_MOD_V23HALF
                                 | V8_MOD_V23
                                 | V8_MOD_V26BIS
                                 | V8_MOD_V26TER
                                 | V8_MOD_V27TER
                                 | V8_MOD_V29
                                 | V8_MOD_V32
                                 | V8_MOD_V34HALF
                                 | V8_MOD_V34
                                 | V8_MOD_V90
                                 | V8_MOD_V92;
    answerer_available_modulations = V8_MOD_V17
                                   | V8_MOD_V21
                                   | V8_MOD_V22
                                   | V8_MOD_V23HALF
                                   | V8_MOD_V23
                                   | V8_MOD_V26BIS
                                   | V8_MOD_V26TER
                                   | V8_MOD_V27TER
                                   | V8_MOD_V29
                                   | V8_MOD_V32
                                   | V8_MOD_V34HALF
                                   | V8_MOD_V34
                                   | V8_MOD_V90
                                   | V8_MOD_V92;

    v8_init(&v8_caller, TRUE, caller_available_modulations, handler, (void *) "caller");
    v8_init(&v8_answerer, FALSE, answerer_available_modulations, handler, (void *) "answerer");
    span_log_set_level(&v8_caller.logging, SPAN_LOG_FLOW | SPAN_LOG_SHOW_TAG);
    span_log_set_tag(&v8_caller.logging, "caller");
    span_log_set_level(&v8_answerer.logging, SPAN_LOG_FLOW | SPAN_LOG_SHOW_TAG);
    span_log_set_tag(&v8_answerer.logging, "answerer");
    for (i = 0;  i < 1000;  i++)
    {
        samples = v8_tx(&v8_caller, amp, SAMPLES_PER_CHUNK);
        if (samples < SAMPLES_PER_CHUNK)
        {
            memset(amp + samples, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - samples));
            samples = SAMPLES_PER_CHUNK;
        }
        remnant = v8_rx(&v8_answerer, amp, samples);
        for (i = 0;  i < samples;  i++)
            out_amp[2*i] = amp[i];
        
        samples = v8_tx(&v8_answerer, amp, SAMPLES_PER_CHUNK);
        if (samples < SAMPLES_PER_CHUNK)
        {
            memset(amp + samples, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - samples));
            samples = SAMPLES_PER_CHUNK;
        }
        if (v8_rx(&v8_caller, amp, samples)  &&  remnant)
            break;
        for (i = 0;  i < samples;  i++)
            out_amp[2*i + 1] = amp[i];

        outframes = afWriteFrames(outhandle,
                                  AF_DEFAULT_TRACK,
                                  out_amp,
                                  samples);
        if (outframes != samples)
        {
            fprintf(stderr, "    Error writing wave file\n");
            exit(2);
        }
    }
    if (afCloseFile(outhandle))
    {
        fprintf(stderr, "    Cannot close wave file '%s'\n", OUTPUT_FILE_NAME);
        exit(2);
    }
    afFreeFileSetup(filesetup);
    
    v8_release(&v8_caller);
    v8_release(&v8_answerer);
    
    if (negotiations_ok != 2)
    {
        printf("Tests failed.\n");
        exit(2);
    }
    
    printf("Tests passed.\n");
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
